## Statystyka praktyczna w data science (R)
## Rozdział 2. Rozkłady danych i prób
# > (c) 2019 Peter C. Bruce, Andrew Bruce, Peter Gedeck

# Importuje wymagane pakiety R.

library(boot)
library(ggplot2)

# Definiuje ścieżki do zestawów danych. Jeżeli przechowujesz dane w innym katalogu, dostosuj ścieżki w kodzie poniżej.

PSDS_PATH <- file.path(dirname(dirname(getwd())))

loans_income <- read.csv(file.path(PSDS_PATH, 'dane', 'loans_income.csv'))[,1]
sp500_px <- read.csv(file.path(PSDS_PATH, 'dane', 'sp500_data.csv.gz'), row.names=1)

# Rysunek 2.1

x <- seq(from=-3, to=3, length=300)
gauss <- dnorm(x)

    par(mar=c(3, 3, 0, 0)+.1)
    plot(x, gauss, type='l', col='blue', xlab='', ylab='', axes=FALSE)
    polygon(x, gauss, col='blue')
dev.off()

    norm_samp <- rnorm(100)
    par(mar=c(3, 3, 0, 0)+.1)
    hist(norm_samp, axes=FALSE, col='red', main='')
dev.off()

## Rozkład próbkowania dla statystyki

# stwórz prostą próbę losową
samp_data <- data.frame(income=sample(loans_income, 1000),
                        type='data_dist')

# stwórz próbę ze średnich z 5 wartości
samp_mean_05 <- data.frame(
  income = tapply(sample(loans_income, 1000*5),
                  rep(1:1000, rep(5, 1000)), FUN=mean),
  type = 'mean_of_5')

# stwórz próbę ze średnich z 20 wartości
samp_mean_20 <- data.frame(
  income = tapply(sample(loans_income, 1000*20),
                  rep(1:1000, rep(20, 1000)), FUN=mean),
  type = 'mean_of_20')

# połącz data.frames i zmień typ na factor
income <- rbind(samp_data, samp_mean_05, samp_mean_20)
income$type <- factor(income$type,
                     levels=c('data_dist', 'mean_of_5', 'mean_of_20'),
                     labels=c('Dane', 'Średnia z 5', 'Średnia z 20'))

ggplot(income, aes(x=income)) +
  geom_histogram(bins=40) +
  facet_grid(type ~ .)

## Próby bootstrapowe
# W obliczeniach wykorzystywane są próby losowe, dlatego wyniki będą różnić się w poszczególnych przebiegach.

stat_fun <- function(x, idx) median(x[idx])
boot_obj <- boot(loans_income, R=1000, statistic=stat_fun)

boot_obj

## Przedziały ufności

# Wersja R rysunku 2.9 jest niedostępna
set.seed(5)
set.seed(7)
sample20 <- sample(loans_income, 20)
sampleMean <- mean(sample20)

stat_fun <- function(x, idx) mean(x[idx])
boot_obj <- boot(sample20, R=500, statistic=stat_fun)
boot_ci <- boot.ci(boot_obj, conf=0.9, type='basic')
X <- data.frame(mean=boot_obj$t)
ci90 <- boot_ci$basic[4:5]
ci <- data.frame(ci=ci90, y=c(9, 11))
# ci <- boot_ci$basic[4:5]
ci
ggplot(X, aes(x=mean)) +
    geom_histogram(bins=40, fill='#AAAAAA') +
    geom_vline(xintercept=sampleMean, linetype=2) +
    geom_path(aes(x=ci, y=10), data=ci, size=2) +
    geom_path(aes(x=ci90[1], y=y), data=ci, size=2) +
    geom_path(aes(x=ci90[2], y=y), data=ci, size=2) +
    geom_text(aes(x=sampleMean, y=20, label='Średnia próby'), size=6) +
    geom_text(aes(x=sampleMean, y=8, label='Przedział 90%'), size=6) +
    theme_bw() + 
    labs(x='', y='Liczba')

## Rozkład normalny
### Standaryzowany rozkład normalny i wykres K-K

norm_samp <- rnorm(100)
qqnorm(norm_samp, main='', xlab='Kwantyl rozkładu normalnego', ylab='z-wynik')
abline(a=0, b=1, col='grey')

## Rozkłady z długimi ogonami

nflx <- sp500_px[,'NFLX']
nflx <- diff(log(nflx[nflx>0]))
qqnorm(nflx)
abline(a=0, b=1, col='grey')

## Rozkład binarny

dbinom(x=2, size=5, p=0.1)

pbinom(2, 5, 0.1)

dbinom(x=0, size=200, p=0.02)

## Rozkład Poissona i jego pochodne
### Rozkład Poissona

rpois(100, lambda=2)

### Rozkład wykładniczy

rexp(n=100, rate=.2)

###  Rozkład Weibulla

rweibull(100, 1.5, 5000)
